import boto3
import botocore.exceptions
import datetime
import json
import global_value as GV
import psycopg2_compf
from error_function import error_handling
import inspect

# 計算モード
CALC_MODE = 'MONTHLY'
# 利用量を取得するLambda
LAMBDA_EXECUTIONLOGCHARGE = GV.ENV.lower() + '-lam-simsp-executionLogCharge'
# 利用料金を計算するLambda
LAMBDA_BILLINGCALCULATION = GV.ENV.lower() + '-lam-sys-billing-calculation'
BUCKET_NAME = GV.ENV.lower() + '-s3-bt-comn-pf-001'
s3_base_path = 'billing-infrastructure/charges/Daily'
# 月額固定（利用料無料）であることを示すラベル
FIXED_MONTHLY_OPTION = '利用料無料'

def lambda_handler(event, context):
    
    print(event)

    try:
        contract_id = None
        
        if 'contract_id' in event:
            contract_id = event['contract_id']
            
        print(contract_id)

        usage_amount = 0
        fixed_amount = 0
        total_amount = 0
        target_file_list = []
        usage_data_list = []
        # 今日の日付を取得
        target_date = datetime.datetime.now()
        # YYYY-MM-DD形式でフォーマット
        formatted_date = target_date.strftime("%Y-%m-%d")
        s3 = boto3.resource('s3')
        setting_bucket = s3.Bucket(BUCKET_NAME)

        if CALC_MODE == 'MONTHLY':
            # 月初から今日までの利用量を取得
            print('月初から今日までの利用量を取得開始')

            response = boto3.client('lambda').invoke(
                FunctionName=LAMBDA_EXECUTIONLOGCHARGE,
                InvocationType='RequestResponse',
                Payload=json.dumps({'target_date' : formatted_date, 'calc_mode' : CALC_MODE})
            )
            if 'StatusCode' in response and response['StatusCode'] == 200:
                # レスポンス読出し
                Payload = json.loads(response['Payload'].read())
                print("---03: body:")
                print(Payload)
            
            print('月初から今日までの利用量を取得完了')

            print('月初から今日までの利用料金を計算開始')
            contractInfo = getContract(contract_id)
            path = create_s3_key(target_date, s3_base_path + '/' + contractInfo['system_id'])
            prefix = path + formatted_date.replace('-', '')
            file_list = setting_bucket.objects.filter(Prefix=prefix)
            
            for o in file_list:            
                target_csv_file = o.key
                target_file_list.append(target_csv_file)

            response = boto3.client('lambda').invoke(
                FunctionName=LAMBDA_BILLINGCALCULATION,
                InvocationType='RequestResponse',
                Payload=json.dumps(
                    {
                        'body' : {
                            "saveToDB": False,
                            "target_date": formatted_date,
                            "target_files": target_file_list
                        }
                    })
            )
            if 'StatusCode' in response and response['StatusCode'] == 200:
                # レスポンス読出し
                Payload = json.loads(response['Payload'].read())
                print("---03: body:")
                print(Payload)
                calc_data = json.loads(Payload['body'])
                for data in calc_data:
                    if data['contract_id'] == contract_id and ('charge_flg'not in data or data['charge_flg']):
                        usage_amount += int(data['amount'])
                        usage_data_list.append(data)
                print('今月の利用料金（従量）：' + str(usage_amount))
            else:
                raise '月初から今日までの利用料金計算に失敗しました'
            print('月初から今日までの利用料金を計算終了')
        elif CALC_MODE == 'DAILY':
            pass

        print('今月の利用料金計算開始')
        fixed_amount, has_fixed_monthly_option = getfixedMonthlyAmount(contract_id)
        print('今月の利用料金（定額）：' + str(fixed_amount))
        total_amount = usage_amount + fixed_amount
        print('今月の利用料金（合計）：' + str(total_amount))
        print('今月の利用料金計算終了')

        # 利用量CSVファイルを削除
        deleteTargetFiles(target_file_list)

        return {
            'statusCode': 200,
            'body': json.dumps({
                'total_amount': total_amount, 
                'has_fixed_monthly_option': has_fixed_monthly_option, 
                'usage_data': usage_data_list})
        }

    except Exception as e:
        print('今月の利用料金の取得に失敗しました。')
        print(e.args)
        print(GV.ENV.lower())
        # 利用量CSVファイルを削除
        deleteTargetFiles(target_file_list)
        return error_handling(inspect.currentframe().f_code.co_name, e, GV.TARGET_DATE)
        
# 利用量CSVファイルが存在するファイルパスの作成
def create_s3_key(d_today :datetime, path: str):
    # date today
    str_year = str(d_today)[0:4]
    str_month = str(d_today)[5:7]
    str_key = path + '/year=' + str_year + '/month=' + str_month + '/'
    # str_key = KEY + '/year=2021/month=07/'
    print(str_key)
    return str_key

# 契約IDから契約情報を取得
def getContract(contract_id):
    try:
        sql = '''
            SELECT
                mc.company_code,
                mc.contract_id,
                tmr.system_id,
                msys.system_name,
                tmr.service_id,
                msrv.service_name
            FROM user_infrastructure.m_contract mc
            JOIN user_infrastructure.t_manage_request tmr ON tmr.contract_id = mc.contract_id
            JOIN billing_infrastructure.m_systems msys ON msys.system_id = tmr.system_id
            JOIN billing_infrastructure.m_services msrv ON msrv.service_id = tmr.service_id
            WHERE
                mc.contract_id = %(contract_id)s
            AND
                ((mc.subscription_start_date < now() AND mc.subscription_end_date IS NULL)
                OR (mc.subscription_start_date < now() AND mc.subscription_end_date > now()))
        '''
        rows = psycopg2_compf.select(sql, {'contract_id' : contract_id})
        print(rows)
        return rows[0]
    except Exception as e:
        print('契約情報の取得に失敗しました。')
        print(e.args)

# 月額固定料金を取得
def getfixedMonthlyAmount(contract_id):
    try:
        sql = '''
            SELECT 
                com.company_code,
                con.contract_id,
                userprod.contract_product_id,
                userprod.quantity,
                CASE
                WHEN (((reg.calc_method)::text = '定額'::text) AND ((reg.quantity_text)::text = '有'::text) AND (userprod.quantity IS NOT NULL)) THEN reg.catalog_amount * userprod.quantity
                ELSE reg.catalog_amount
                END AS amount,
                userprod.use_start_date,
                userprod.use_end_date,
                reg.billing_category,
                serv.service_name,
                sys.system_name
            FROM (((((user_infrastructure.m_company com
                JOIN user_infrastructure.m_contract con ON (((com.company_code)::text = (con.company_code)::text)))
                JOIN user_infrastructure.m_contract_product userprod ON (((con.contract_id)::text = (userprod.contract_id)::text)))
                JOIN billing_infrastructure.m_regular_products reg ON (((reg.product_id)::text = (userprod.product_id)::text)))
                JOIN billing_infrastructure.m_services serv ON (((reg.service_id)::text = (serv.service_id)::text)))
                JOIN billing_infrastructure.m_systems sys ON (((sys.system_id)::text = (reg.system_id)::text)))
            WHERE (
            ((reg.calc_method)::text = '定額'::text)
            -- 契約期間条件
            AND ((CURRENT_DATE between con.applicable_start_date AND DATE_TRUNC('month', con.applicable_end_date + '1 months') + '-1 days') 
            OR ((CURRENT_DATE >= con.applicable_start_date) AND con.applicable_end_date IS NULL))
            -- 契約商品の期間指定
            AND ((CURRENT_DATE between userprod.use_start_date and userprod.use_end_date)
            OR (CURRENT_DATE >= userprod.use_start_date AND userprod.use_end_date IS NULL))
            -- 通常商品の期間指定
            AND ((CURRENT_DATE between reg.applicable_start_date and reg.applicable_end_date)
            OR (CURRENT_DATE >= reg.applicable_start_date AND reg.applicable_end_date IS NULL)))
            AND con.contract_id = %(contract_id)s
            ORDER BY com.company_code, userprod.contract_product_id
        '''
        rows = psycopg2_compf.select(sql, {'contract_id' : contract_id})
        print(rows)
        used_amounts = 0
        has_fixed_monthly_option = False
        for row in rows:
            if row['billing_category'] == FIXED_MONTHLY_OPTION:
                has_fixed_monthly_option = True
            used_amounts += int(row['amount'])
        return used_amounts, has_fixed_monthly_option
    except Exception as e:
        print('月額固定料金の取得に失敗しました。')
        print(e.args)

# S3のファイル削除
def deleteTargetFiles(target_file_list):
    for file in target_file_list:
        try:
            boto3.client('s3').delete_object(Bucket=BUCKET_NAME, Key=file)
        except botocore.exceptions.ClientError as e:
            "FILE NOT FOUND."
