from flask import Flask, Blueprint, render_template, current_app, make_response
from flask import send_file, Response
from src.component.common.entraid.parameters import *
from src.component.decorator.authenticate import *
from src.component.model.simsp import mscone_lic as sml
from src.component.model.tables import *
from src.component.common.pdbc import pdbc
from src.component.common.pdbc import db_hosts
import uuid
import os
import time
import shutil
import sys
import math
import requests
import tempfile
from io import StringIO
from netapp_ontap import NetAppRestError
from netapp_ontap import HostConnection
from netapp_ontap.resources import FileInfo, FileCopy, QuotaReport

CONNECTION_TIMEOUT = 300
READ_TIMEOUT = 300

def get_env_id(company_code):
    company_condition = sml.MCompany()
    company_condition.company_code = company_code
    return pdbc.select(db_hosts.simsp(), company_condition)[0]['env_id']

#####
# 2024/10/02 Phase6対応
# ログファイル一覧を取得する
# Create: nakano
#####
def get_loglist(company_code, names):
    loglist = []
    dir_list = []
    env_id = get_env_id(company_code)
    if len(names) == 0:
        return loglist
    
    for name in names:
        prefix = os.path.dirname(name)
        if not prefix in dir_list:
            dir_list.append(prefix)
    try:
        with HostConnection(
                current_app.config['ONTAP'], 
                username=current_app.config['ONTAP_USER'], 
                password=current_app.config['ONTAP_USER_PASSWORD'], 
                verify=False
            ):
            for dir in dir_list:
                files = FileInfo.get_collection(
                    current_app.config['ONTAP_VOLUME_UUID'], 
                    env_id + '/' + dir,
                    type="file",
                    fields="name,type,size,changed_time"
                    )
                for file_info in files:
                    if dir + '/' + file_info.name in names:
                        file_obj = {}
                        file_obj['filename'] = dir + '/' + file_info.name
                        file_obj['size'] = file_info.size
                        file_obj['date'] = file_info.changed_time.strftime('%Y-%m-%d')
                        file_obj['time'] = file_info.changed_time.strftime('%H:%M:%S')
                        loglist.append(file_obj)
        return loglist
    except Exception as e:
        print(e)
        raise Exception("")

#####
# 2024/10/02 Phase6対応
# ログファイルを取得する
# Create: nakano
#####
def get_text(company_code, filepath):
    env_id = get_env_id(company_code)
    res_str = ""
    try:
        with HostConnection(
                current_app.config['ONTAP'], 
                current_app.config['ONTAP_USER'], 
                password=current_app.config['ONTAP_USER_PASSWORD'], 
                verify=False
            ) as conn:
            resource = FileInfo(current_app.config['ONTAP_VOLUME_UUID'], env_id + '/' + filepath)
            resource.get(return_metadata=True)
            file_size = resource.size
            # サイズが大きすぎるとエラーになるので100KBずつ分割ダウンロード
            offset = 0
            chunk_size=1024*100
            conn.request_headers = {"Accept": "multipart/form-data"}
            resource = FileInfo(current_app.config['ONTAP_VOLUME_UUID'], env_id + '/' + filepath)
            while offset < file_size:
                resource.get(byte_offset=offset, length=chunk_size)
                res_str += resource.response_files[0]['content'].decode()
                offset += chunk_size
            return res_str
    except Exception as e:
        print(e)
        return ''
    
#####
# 2024/10/02 Phase6対応
# ログファイルの最新から100行を取得する
# Create: nakano
#####
def get_text_limit(company_code, filepath):
    env_id = get_env_id(company_code)
    lines = []
    res_str = ""
    tmp_str = ""
    try:
        with HostConnection(
                current_app.config['ONTAP'], 
                current_app.config['ONTAP_USER'], 
                password=current_app.config['ONTAP_USER_PASSWORD'], 
                verify=False
            ) as conn:

            resource = FileInfo(current_app.config['ONTAP_VOLUME_UUID'], env_id + '/' + filepath)
            resource.get(return_metadata=True)
            file_size = resource.size
            # サイズが大きすぎるとエラーになるので100KBずつ分割ダウンロード
            offset = 0
            chunk_size=1024*100
            conn.request_headers = {"Accept": "multipart/form-data"}
            resource = FileInfo(current_app.config['ONTAP_VOLUME_UUID'], env_id + '/' + filepath)
            while offset < file_size:
                resource.get(byte_offset=offset, length=chunk_size)
                tmp_str += resource.response_files[0]['content'].decode()
                offset += chunk_size
            with StringIO(tmp_str) as sys.stdin:
                lines = sys.stdin.readlines()
            if len(lines) > 100:
                res_str = "".join(lines[-100:])
            else:
                res_str = "".join(lines)
            return res_str
    except Exception as e:
        print(e)
        return ''

#####
# 2024/10/02 Phase6対応
# ファイル・フォルダの一覧を取得する
# Create: nakano
#####
def get_list(company_code, prefix):
    env_id = get_env_id(company_code)
    print('env_id = ' + env_id)

    file_list = []
    folder_list = []

    try:
        print("--------------MODE FSx ONTAP--------------")
        #print(get_size(company_code, '/'))
        with HostConnection(
                current_app.config['ONTAP'], 
                username=current_app.config['ONTAP_USER'], 
                password=current_app.config['ONTAP_USER_PASSWORD'], 
                verify=False
            ):
            files = FileInfo.get_collection(
                current_app.config['ONTAP_VOLUME_UUID'], 
                env_id + '/' + prefix,
                type="file|directory",
                fields="name,type,size,changed_time"
                )
            for file_info in files:
                if file_info.name not in (".", ".."):
                    print(file_info.to_dict())
                    if file_info.type == "file":
                        file_obj = {}
                        print('ファイル ----------------------------')
                        file_obj['name'] = file_info.name
                        file_obj['size'] = file_info.size
                        file_obj['date'] = file_info.changed_time.strftime('%Y-%m-%d %H:%M:%S')
                        file_list.append(file_obj)
                    elif file_info.type == "directory" and file_info.name != ".snapshot":
                        folder_obj = {}
                        print('フォルダー ----------------------------')
                        folder_obj['name'] = file_info.name
                        folder_obj['size'] = file_info.size
                        folder_obj['date'] = file_info.changed_time.strftime('%Y-%m-%d %H:%M:%S')
                        folder_list.append(folder_obj)
    except Exception as e:
        print(e)
        pass

    list = {}
    list['folders'] = folder_list
    list['files'] = file_list
    data = []
    for folder in list['folders']:
        item = {}
        item['name'] = folder['name']
        item['cal_start'] = ''
        item['cal_end'] = ''
        item['status'] = 0
        item['last_modified'] = ''
        item['size'] = ''
        item['is_folder'] = True
        data.append(item)

    for file in list['files']:
        item = {}
        item['name'] = file['name']
        item['cal_start'] = ''
        item['cal_end'] = ''
        item['status'] = 0
        item['last_modified'] = file['date']
        item['size'] = file['size']
        item['is_folder'] = False
        data.append(item)
    return json.dumps(data)

#####
# 2024/10/02 Phase6対応
# フォルダを作成する
# Create: nakano
#####
def make_dir(company_code, prefix):
    """Create a directory on a volume"""
    env_id = get_env_id(company_code)
    pathname = env_id + '/' + prefix
    try:
        with HostConnection(
                current_app.config['ONTAP'], 
                username=current_app.config['ONTAP_USER'], 
                password=current_app.config['ONTAP_USER_PASSWORD'], 
                verify=False
            ):
            resource = FileInfo(current_app.config['ONTAP_VOLUME_UUID'], pathname)
            resource.type = "directory"
            resource.unix_permissions = "644"
            resource.post(hydrate=True)
            print(resource)
    except NetAppRestError as error:
        print(error)
    return 

#####
# 2024/10/02 Phase6対応
# FSxにファイルをアップロードする
# サイズが大きいファイルは分割アップロードする
# Create: nakano
#####
def create_file(pathname, contents):
    """Create a file on a volume"""
    try:
        with HostConnection(
                current_app.config['ONTAP'],
                username=current_app.config['ONTAP_USER'],
                password=current_app.config['ONTAP_USER_PASSWORD'],
                verify=False,
                headers={"Accept": "multipart/form-data"},
                protocol_timeouts=(CONNECTION_TIMEOUT, READ_TIMEOUT)
            ):
            resource = FileInfo(current_app.config['ONTAP_VOLUME_UUID'], pathname)
            count = 1
            chunk_size = 1024 * 1024 # 1 MB
            resource.post(
                data=contents.read(chunk_size),
                overwrite=True,
                byte_offset=-1
            )
            while True:
                chunk = contents.read(chunk_size)
                if not chunk:
                    break
                resource.patch(data=chunk, byte_offset=count*chunk_size)
                count += 1
    except NetAppRestError as error:
        print(error)
        # 空き容量が不足
        if error.status_code == requests.codes.server_error:
            with HostConnection(
                    current_app.config['ONTAP'], 
                    current_app.config['ONTAP_USER'], 
                    current_app.config['ONTAP_USER_PASSWORD'], 
                    verify=False
                ):
                resource = FileInfo(current_app.config['ONTAP_VOLUME_UUID'], pathname)
                resource.delete()
        raise error
    # with HostConnection(
    #     current_app.config['ONTAP'],
    #     username=current_app.config['ONTAP_USER'],
    #     password=current_app.config['ONTAP_USER_PASSWORD'],
    #     verify=False,
    #     protocol_timeouts=(CONNECTION_TIMEOUT, READ_TIMEOUT)
    # ):
    #     resource = FileInfo(current_app.config['ONTAP_VOLUME_UUID'], pathname)
    #     resource.post(
    #         hydrate=True, data=contents)
    #     resource.patch()

#####
# 2024/10/02 Phase6対応
# ファイル・フォルダをアップロードする
# Create: nakano
#####
def upload(company_code, folder_name, request):
    file_cnt = 0
    if request.form.get('path'):
        path_list = request.form.get('path').split('|')
    #ファイルサイズチェック
    size = 0
    for key in request.files:
        file = request.files.get(key)
        pos = file.tell()
        file.seek(0, os.SEEK_END)
        size = (size + file.tell())
        file.seek(pos)
    if 10737418240 < size:
        return Response(status=500, mimetype='application/json')

    env_id = get_env_id(company_code)
    print('env_id = ' + env_id)
    #アップロード処理
    for key in dict(request.files.lists()):
        try:
            upload_key = str(uuid.uuid4())
            dest_path = f"{current_app.config['ONTAP_MOUNT_PATH']}{upload_key}/"
            os.makedirs(dest_path)
            for f in dict(request.files.lists())[key]:
                if request.form.get('path'):
                    file_name = path_list[file_cnt]
                else:
                    file_name = os.path.basename(f.name)
                if file_name.startswith("/"):
                    file_name = file_name[1:]
                if folder_name != '':
                    file_name = folder_name + '/' + file_name
                ##フォルダ存在チェック
                with HostConnection(
                        current_app.config['ONTAP'], 
                        username=current_app.config['ONTAP_USER'], 
                        password=current_app.config['ONTAP_USER_PASSWORD'], 
                        verify=False
                    ):
                    dir_name = os.path.dirname(f"/{env_id}/{file_name}")
                    resource = FileInfo.find(current_app.config['ONTAP_VOLUME_UUID'], os.path.dirname(dir_name), name=os.path.basename(dir_name))
                    if resource is None:
                        make_dir(company_code, os.path.dirname(file_name))
                # create_file('/' + env_id + '/' + file_name, f.stream)
                if not os.path.isdir(os.path.dirname(dest_path + file_name)):
                    os.makedirs(os.path.dirname(dest_path + file_name))
                f.save(dest_path + file_name)
                with HostConnection(
                        current_app.config['ONTAP'], 
                        current_app.config['ONTAP_USER'], 
                        password=current_app.config['ONTAP_USER_PASSWORD'], 
                        verify=False
                    ):
                    resource = FileCopy()
                    resource.files_to_copy = [{
                        "source":{"volume":{"uuid":current_app.config['ONTAP_TEMP_VOLUME_UUID']},"path":f"{upload_key}/{file_name}"},
                        "destination":{"volume":{"uuid":current_app.config['ONTAP_VOLUME_UUID']},"path":f"/{env_id}/{file_name}"}}]
                    resource.post(hydrate=True, poll=False)
                file_cnt += 1
        except Exception as e:
            return Response(status=e.status_code, mimetype='application/json')

    return Response(status=200, mimetype='application/json')

#####
# 2024/10/02 Phase6対応
# ファイル・フォルダをリネームする
# Create: nakano
#####
def rename(company_code, old_name, new_name):
    env_id = get_env_id(company_code)
    print('env_id = ' + env_id)
    if old_name[-1] == '/':
        old_name = old_name[:-1]
    if new_name[-1] == '/':
        new_name = new_name[:-1]
    old_path = (env_id + '/' + old_name)
    new_path = (env_id + '/' + new_name)
    messages = {}
    try:
        with HostConnection(
                current_app.config['ONTAP'], 
                current_app.config['ONTAP_USER'], 
                password=current_app.config['ONTAP_USER_PASSWORD'], 
                verify=False
            ):
            ##同名チェック
            resource = FileInfo.find(current_app.config['ONTAP_VOLUME_UUID'], os.path.dirname(new_path), name=os.path.basename(new_path))
            if resource:
                print(resource)
                messages = {'status': 'ng','msg': 'この名前はすでに使われています。'}
                return json.dumps(messages)
    
            ##名称変更
            resource = FileInfo(current_app.config['ONTAP_VOLUME_UUID'], path=old_path)
            resource.path = new_path
            resource.patch()
        messages = {'status': 'ok','msg': 'ファイル名の更新が完了しました。'}

    except Exception as e:
        print(e)

    return json.dumps(messages)

#####
# 2024/10/02 Phase6対応
# ファイル・フォルダを削除する
# Create: nakano
#####
def delete(company_code, folder_name, prefixes):
    env_id = get_env_id(company_code)
    """Delete a file or directory on a volume"""
    try:
        with HostConnection(
                current_app.config['ONTAP'], 
                current_app.config['ONTAP_USER'], 
                current_app.config['ONTAP_USER_PASSWORD'], 
                verify=False
            ):
            for del_prefix in prefixes:
                is_directory = del_prefix[-1] == '/'
                if not folder_name:
                    pathname = '/' + env_id + '/' + del_prefix
                else:
                    pathname = '/' + env_id + '/' + folder_name + '/' + del_prefix
                resource = FileInfo(current_app.config['ONTAP_VOLUME_UUID'], path=pathname)
                resource.delete(recurse=is_directory)
    except NetAppRestError as error:
        print(error)

    return 'Okey!'

#####
# 2024/10/02 Phase6対応
# ファイル・フォルダをダウンロードする
# 複数ファイルの場合は圧縮する
# Create: nakano
#####
def download(company_code, prefix, items):
    try:
        print("##################ダウンロード開始##################")
        processing_start_time = time.time()  # 開始時刻を記録

        env_id = get_env_id(company_code)
        print('env_id = ' + env_id)
        download_key = str(uuid.uuid4())
        dest_path = f"{current_app.config['ONTAP_MOUNT_PATH']}{download_key}/download/"
        os.makedirs(dest_path)
        print("################## 1 ##################")
        with HostConnection(
                current_app.config['ONTAP'], 
                current_app.config['ONTAP_USER'], 
                password=current_app.config['ONTAP_USER_PASSWORD'], 
                verify=False, 
                protocol_timeouts=(CONNECTION_TIMEOUT, READ_TIMEOUT)
            ):
            copy_list = []
            for item in items:
                if item['is_folder']:
                    os.makedirs(dest_path + item['name'])
                    file_list = list_files(f"/{env_id}/{prefix}/{item['name']}")
                    for file in file_list:
                        # コピー元のファイルパスの不要部分を削除
                        idx = file.path.find(env_id)
                        file_path = file.path[idx+len(env_id)+1:]
                        if prefix:
                            idx = file.path.find(prefix)
                            file_path = file.path[idx+len(prefix)+1:]
                        if file.type == 'file':
                            copy_list.append(
                                {"source":{"volume":{"uuid":current_app.config['ONTAP_VOLUME_UUID']},"path":f"{file.path}/{file.name}"},
                                "destination":{"volume":{"uuid":current_app.config['ONTAP_TEMP_VOLUME_UUID']},"path":f"/{download_key}/download/{file_path}/{file.name}"}})
                        elif file.type == 'directory':
                            os.makedirs(f"{current_app.config['ONTAP_MOUNT_PATH']}{download_key}/download/{file_path}/{file.name}")
                else:
                    copy_list.append(
                        {"source":{"volume":{"uuid":current_app.config['ONTAP_VOLUME_UUID']},"path":f"/{env_id}/{prefix}/{item['name']}"},
                        "destination":{"volume":{"uuid":current_app.config['ONTAP_TEMP_VOLUME_UUID']},"path":f"/{download_key}/download/{item['name']}"}})
            if len(copy_list) > 0:
                # 10ファイルずつコピー
                for i in range(0, len(copy_list), 10):
                    resource = FileCopy()
                    resource.files_to_copy = copy_list[i:i + 10]
                    print("################## 2 ##################")

                    resource.post(hydrate=True, poll=False)

                    print("################## 3 ##################")

                    # コピー処理が完了するまで待機
                    start_time = time.time()
                    max_wait_time = 600  # 最大待機時間 10分 (600秒)
                    while True:
                        all_copied = True
                        for file_copy in resource.files_to_copy:
                            destination_path = file_copy['destination']['path']
                            print('file_path: ' + f"{current_app.config['ONTAP_MOUNT_PATH']}{destination_path}")
                            if not os.path.exists(f"{current_app.config['ONTAP_MOUNT_PATH']}{destination_path}"):
                                all_copied = False
                                break

                        if all_copied:
                            break
                        if time.time() - start_time >= max_wait_time:
                            raise TimeoutError("ファイルコピーの待機時間が超過しました。")
                        time.sleep(5)  # 5秒ごとにチェック

        # send_file呼び出し前までの処理時間を計測
        elapsed_time = time.time() - processing_start_time
        print(f"ファイルコピー処理にかかった時間: {elapsed_time:.2f}秒")  # 時間をコンソールに出力

        if len(items) == 1 and not items[0]['is_folder']:
            #単一ファイルの場合
            ret = send_file(dest_path + str(items[0]['name']))
            print("##################ダウンロード終了##################")
            return ret
        else:
            # 複数ファイルの場合、圧縮処理の時間を計測
            archive_start_time = time.time()  # 圧縮処理開始時刻を記録
            shutil.make_archive(f"{current_app.config['ONTAP_MOUNT_PATH']}{download_key}/archive", format='zip', root_dir=f"{current_app.config['ONTAP_MOUNT_PATH']}{download_key}/download")
            archive_elapsed_time = time.time() - archive_start_time  # 圧縮処理の経過時間を計算
            print(f"アーカイブ作成にかかった時間: {archive_elapsed_time:.2f}秒")  # 時間をコンソールに出力

            shutil.rmtree(f"{current_app.config['ONTAP_MOUNT_PATH']}{download_key}/download")
            ret = send_file(f"{current_app.config['ONTAP_MOUNT_PATH']}{download_key}/archive.zip", as_attachment=False, attachment_filename='download.zip')
            print("##################ダウンロード終了##################")
            return ret

    except Exception as e:
        print(e)
    return

#####
# 2024/10/02 Phase6対応
# 企業のストレージ容量を取得する
# Create: nakano
#####
def get_size(company_code, dname):
    env_id = get_env_id(company_code)
    print('env_id = ' + env_id)
    try:
        with HostConnection(
                current_app.config['ONTAP'], 
                username=current_app.config['ONTAP_USER'], 
                password=current_app.config['ONTAP_USER_PASSWORD'], 
                verify=False
            ):
            #print(list(QuotaReport.get_collection(type='tree')))
            for report in QuotaReport.get_collection(type='tree'):
                resource = QuotaReport(
                    index=report.index, **{"volume.uuid": report.volume.uuid}
                )
                resource.get()
                #print(resource)
                if env_id == resource['qtree']['name']:
                    size = []
                    size.append(convert_size(resource['space']['used']['total'])) # 使用量 10M
                    size.append(convert_size(resource['space']['hard_limit'])) # 総容量 100M
                    size.append(f"{resource['space']['used']['hard_limit_percent']} %") # 使用% 10%
                    return size
    except Exception as e:
        print(e)
#####
# 2024/10/02 Phase6対応
# FSxからファイルを取得する
# Create: nakano
#####
def get_file(company_code, file_path):
    try:
        env_id = get_env_id(company_code)
        print('env_id = ' + env_id)
        download_key = str(uuid.uuid4())
        dest_path = f"{current_app.config['ONTAP_MOUNT_PATH']}{download_key}/download/"
        os.makedirs(dest_path)
        with HostConnection(
                current_app.config['ONTAP'], 
                current_app.config['ONTAP_USER'], 
                password=current_app.config['ONTAP_USER_PASSWORD'], 
                verify=False
            ):
            resource = FileCopy()
            resource.files_to_copy = [{
                "source":{"volume":{"uuid":current_app.config['ONTAP_VOLUME_UUID']},"path":f"/{env_id}/{file_path}"},
                "destination":{"volume":{"uuid":current_app.config['ONTAP_TEMP_VOLUME_UUID']},"path":f"/{download_key}/download/{os.path.basename(file_path)}"}}]
            resource.post(hydrate=True, poll=False)
            time.sleep(1)
    except Exception as e:
        print(e)
    return dest_path + os.path.basename(file_path)

#####
# 2024/10/02 Phase6対応
# ファイル・フォルダの一覧を再帰的に取得する
# Create: nakano
#####
def list_files(path, file_list=None):
    if not file_list:
        file_list = []
    files = FileInfo.get_collection(current_app.config['ONTAP_VOLUME_UUID'], path)
    for file_info in files:
        if file_info.name not in (".", ".."):
            if file_info.type == "file":
                file_list.append(file_info)
            elif file_info.type == "directory" and file_info.name != ".snapshot":
                file_list.append(file_info)
                list_files(f"{file_info.path}/{file_info.name}/", file_list)
    
    return file_list

#####
# 2024/10/02 Phase6対応
# バイトを適切な単位に変換する
# Create: nakano
#####
def convert_size(size):
    units = ("B", "KB", "MB", "GB", "TB", "PB", "EB", "ZB")
    i = math.floor(math.log(size, 1024)) if size > 0 else 0
    size = round(size / 1024 ** i, 2)

    return f"{size} {units[i]}"